<?php
/**
 * Plugin Name: 簡易株価表示ツール
 * Plugin URI: https://tokugawax.jp/chart-wordpress-plugin/
 * Description: 指定した銘柄の株価・チャートを表示するためのプラグインです。チャートの非表示にも対応しています。商用利用はお控え下さい。
 * Version: 1.0.0
 * Author: tokugawa
 * Author URI: https://tokugawax.jp/
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Chart.js の読み込み
function ssc_enqueue_scripts() {
    wp_enqueue_script( 'chart-js', 'https://cdn.jsdelivr.net/npm/chart.js', array(), null, true );
}
add_action( 'wp_enqueue_scripts', 'ssc_enqueue_scripts' );

// ショートコード [stock_chart symbol="AAPL" name="Apple" period="1mo" ...]
function ssc_custom_stock_chart_shortcode( $atts ) {
    // 【WP制御要件】WordPress側で制御するコマンドのデフォルト値
    $atts = shortcode_atts( array(
        'symbol'      => '7203.T',      // 1. 表示させたい銘柄の証券コード/ティッカー
        'name'        => 'トヨタ自動車', // 銘柄名
        'period'      => '1mo',         // 2. チャート表示期間 (1d, 1wk, 1mo, 3mo, 6mo, 1y, 5y, 10y)
        'color'       => '#2962FF',     // 3. チャートの色
        'show_chart'  => 'true',        // 4. チャートの表示・非表示 (true/false)
        'show_symbol' => 'true',        // 5. 証券コードの表示・非表示 (true/false)
        'show_name'   => 'true',        // 6. 銘柄名の表示・非表示 (true/false)
        'unit'        => 'auto'         // 通貨単位 (auto, none, $, 円 など指定可能)
    ), $atts, 'stock_chart' );

    $symbol = esc_attr( $atts['symbol'] );
    $name   = esc_attr( $atts['name'] );
    $period = esc_attr( $atts['period'] );
    $color  = esc_attr( $atts['color'] );
    $unit   = esc_attr( $atts['unit'] );
    
    // 文字列の 'true' / 'false' を真偽値に変換
    $show_chart  = filter_var( $atts['show_chart'], FILTER_VALIDATE_BOOLEAN );
    $show_symbol = filter_var( $atts['show_symbol'], FILTER_VALIDATE_BOOLEAN );
    $show_name   = filter_var( $atts['show_name'], FILTER_VALIDATE_BOOLEAN );
    
    $unique_id = uniqid();
    $container_id = 'ssc-container-' . $unique_id;
    $canvas_id = 'ssc-chart-' . $unique_id;

    ob_start();
    ?>
    <div id="<?php echo $container_id; ?>" class="ssc-container" style="max-width: 800px; margin: 20px auto; font-family: sans-serif; border: 1px solid #eee; padding: 15px; border-radius: 5px; background: #fff;">
        
        <?php if ( $show_symbol || $show_name ) : ?>
            <h3 class="ssc-title" style="margin-top: 0; margin-bottom: 10px; display: flex; align-items: baseline; gap: 10px;">
                <?php if ( $show_symbol ) : ?>
                    <span class="ssc-symbol"><?php echo $symbol; ?></span>
                <?php endif; ?>
                <?php if ( $show_name && !empty($name) ) : ?>
                    <span class="ssc-name" style="font-size: 0.8em; color: #555;"><?php echo $name; ?></span>
                <?php endif; ?>
            </h3>
        <?php endif; ?>
        
        <div class="ssc-price-info" style="margin-bottom: 15px;">
            <span style="font-size: 28px; font-weight: bold;" class="current-price">読込中...</span>
            <span style="font-size: 16px; margin-left: 12px; font-weight: bold;" class="change-info"></span>
        </div>

        <?php if ( $show_chart ) : // 【WP制御要件4】チャート表示がtrueの場合のみ出力 ?>
            <div class="ssc-period-dropdown" style="margin-bottom: 15px; text-align: right;">
                <select class="ssc-select-period" style="padding: 6px 12px; border-radius: 4px; border: 1px solid #ccc; font-size: 14px; cursor: pointer;">
                    <?php 
                    $periods = [
                        '1d'  => '1日', 
                        '1wk' => '1週間', 
                        '1mo' => '1か月', 
                        '3mo' => '3か月', 
                        '6mo' => '6か月', 
                        '1y'  => '1年', 
                        '5y'  => '5年', 
                        '10y' => '10年'
                    ];
                    foreach ($periods as $val => $label) {
                        $selected = ($val === $period) ? 'selected' : '';
                        echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
                    }
                    ?>
                </select>
            </div>

            <div style="position: relative; height: 300px; width: 100%;">
                <canvas id="<?php echo $canvas_id; ?>" data-symbol="<?php echo $symbol; ?>" data-period="<?php echo $period; ?>" data-color="<?php echo $color; ?>"></canvas>
            </div>
        <?php endif; ?>

        <div class="ssc-credit" style="text-align: right; font-size: 10px; margin-top: 15px; color: #888;">
            Powered by <a href="https://tokugawax.jp/" target="_blank" rel="noopener" style="color: #888; text-decoration: none;">PayINT</a>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const container = document.getElementById('<?php echo $container_id; ?>');
        const symbol = '<?php echo $symbol; ?>';
        const initialPeriod = '<?php echo $period; ?>';
        const chartColor = '<?php echo $color; ?>';
        const showChart = <?php echo $show_chart ? 'true' : 'false'; ?>;
        
        let canvas = null;
        let stockChart = null;

        if (showChart) {
            canvas = document.getElementById('<?php echo $canvas_id; ?>');
        }

        // チャートを描画する関数
        function renderChart(dates, prices) {
            if (!showChart || !canvas) return;

            const ctx = canvas.getContext('2d');
            if (stockChart) {
                stockChart.destroy();
            }

            stockChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: dates, // 横軸（時間または日付）
                    datasets: [{
                        data: prices,
                        borderColor: chartColor, // 【WP制御要件3】色
                        backgroundColor: chartColor + '33',
                        borderWidth: 2,
                        pointRadius: 0,
                        pointHoverRadius: 4,
                        fill: true,
                        tension: 0.1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false } // 凡例非表示
                    },
                    scales: {
                        x: { display: true, ticks: { maxTicksLimit: 10 } },
                        y: { display: true }
                    }
                }
            });
        }

        // 株価と増減を画面に反映する関数
        function updatePriceDisplay(currentPrice, prevPrice) {
            const currentEl = container.querySelector('.current-price');
            const changeEl = container.querySelector('.change-info');
            
            const changeAmount = currentPrice - prevPrice;
            const changePercent = prevPrice !== 0 ? (changeAmount / prevPrice) * 100 : 0;
            
            // 【Web表示要件3】増加は赤字、減少は緑字
            let colorCode = '#333';
            let sign = '';
            
            if (changeAmount > 0) {
                colorCode = '#d32f2f'; // 増加: 赤字
                sign = '+';
            } else if (changeAmount < 0) {
                colorCode = '#388e3c'; // 減少: 緑字
                sign = '-';
            }

            // 【Web表示要件6】通貨単位の自動判定ロジック
            let prefix = '';
            let suffix = '';
            let unitSetting = '<?php echo esc_js($unit); ?>';
            
            if (unitSetting === 'auto') {
                // S&P500(^GSPC)や日経平均(^N225)など、^で始まるものは指数とみなす
                if (symbol.startsWith('^') || symbol.toUpperCase().includes('INDEX')) {
                    // 単位なし
                } else if (symbol.endsWith('.T')) {
                    // 末尾が .T の場合は日本株
                    suffix = '円';
                } else {
                    // それ以外は米国株扱いとして $ を付ける
                    prefix = '$';
                }
            } else if (unitSetting === 'none' || unitSetting === '') {
                // 手動で単位なしを指定した場合
            } else if (unitSetting === '$') {
                prefix = '$';
            } else {
                suffix = unitSetting;
            }

            // 価格をフォーマットする内部関数
            function formatVal(val) {
                let absVal = Math.abs(val);
                // 少数第2位まで表示 (不要な0は省く)
                let formatted = absVal.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
                return prefix + formatted + (suffix ? ' ' + suffix : '');
            }

            currentEl.innerText = formatVal(currentPrice);
            changeEl.innerText = `${sign}${formatVal(changeAmount)} (${sign}${Math.abs(changePercent).toFixed(2)}%)`;
            changeEl.style.color = colorCode;
        }

        // データを取得して画面を更新する関数
        async function fetchAndRenderData(currentPeriod) {
            // Renderで構築したAPIのURL
            const baseUrl = 'https://my-stock-api-xlb3.onrender.com/stock';
            const apiUrl = `${baseUrl}?symbol=${symbol}&period=${currentPeriod}`;
            
            try {
                const response = await fetch(apiUrl);
                const data = await response.json();
                
                if (data.error) {
                    console.error('API Error:', data.error);
                    container.querySelector('.current-price').innerText = 'データ取得エラー';
                    return;
                }

                updatePriceDisplay(data.currentPrice, data.prevPrice);

                if (showChart) {
                    renderChart(data.dates, data.prices);
                }

            } catch (error) {
                console.error('Data fetch error:', error);
                container.querySelector('.current-price').innerText = '通信エラー';
            }
        }

        // 初回実行
        fetchAndRenderData(initialPeriod);

        // 【Web表示要件1-1】プルダウン変更のイベントリスナー
        if (showChart) {
            const selectDropdown = container.querySelector('.ssc-select-period');
            if (selectDropdown) {
                selectDropdown.addEventListener('change', function() {
                    const selectedPeriod = this.value;
                    fetchAndRenderData(selectedPeriod);
                });
            }
        }
    });
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode( 'stock_chart', 'ssc_custom_stock_chart_shortcode' );